/*****************************************************************************
*  LPC54x WS2811 LED driver demo LPC15xx board
*
*  1. Use SCT timer to transmit WS2811 LED driver frames (24 bit)
*  2. SCT_OUT5 is linked to P1_2
*****************************************************************************/

#include "board.h"
#include "sct_ws2811.h"

/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/

/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/
volatile int busy;
volatile int count;

const uint16_t pattern[] =
{
    0x123, 0x456,                                      // 24 bit WS2811 frames split into 2 x 12 bits
    0xFF0, 0x0CC,
    0x555, 0x555,
    0x800, 0x001,
};
/*****************************************************************************
 * Private functions
 ****************************************************************************/

/*****************************************************************************
 * Public functions
 **********************************************************/

void LEDDRIVER_open(void)
{

	Chip_SCT_Init(LPC_SCT);

	LPC_SCT->STATE_L         = 9;                        // start state
	LPC_SCT->OUTPUT         &= ~((1 << DATA_OUT)          // preset data output low
	                              |   (1 << AUX_OUT));     // preset aux output low

	Chip_SCT_SetMatchReload(LPC_SCT, SCT_MATCH_0, (WS2811_FB - 1));	// full bit period
	Chip_SCT_SetMatchReload(LPC_SCT, SCT_MATCH_1, (WS2811_T0H - 1));	// preset data output low
	Chip_SCT_SetMatchReload(LPC_SCT, SCT_MATCH_2, (WS2811_T1H - 1));	// preset aux output low

	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_12, ENABLE_ALL_STATES);	 		// happens in all states
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_12, (CHIP_SCT_EVENTCTRL_T)  ( SCT_EVECTRL_MATCH0 	|	  // related to MATCH0_L
                                                                          SCT_COMBMODE_MATCH	|	  // match only
                                                                          SCT_STATELD_0			  |
                                                                          (31 << 15)			    ));  // add value to STATE



	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_11, 0x00000FFE);	 			// all data bit states except state 0
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_11, (CHIP_SCT_EVENTCTRL_T)  ( SCT_EVECTRL_MATCH2 	|	// related to MATCH2_L
                                                                          SCT_COMBMODE_MATCH	));	// match only

	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_10, 0x00000FFF);	 			// all data bit states
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_10, (CHIP_SCT_EVENTCTRL_T)  ( SCT_EVECTRL_MATCH1 	|	// related to MATCH1_L
                                                                          SCT_COMBMODE_MATCH	));	// match only

	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_9, 0);	 						// contains the 12 bits written by the app
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_9, (CHIP_SCT_EVENTCTRL_T)  ( SCT_EVECTRL_MATCH1 	|	// related to MATCH1_L
                                                                          SCT_OUTSEL_H			  |	// use OUTPUT for I/O condition
                                                                          (AUX_OUT << 6)		  |	// Use AUX output signal
                                                                          SCT_IOCOND_LOW		  |	// AUX = 0
                                                                          SCT_COMBMODE_AND		));	// match AND I/O

	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_8, 0);	 						// contains the 12 bits written by the app
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_8, (CHIP_SCT_EVENTCTRL_T)  ( SCT_EVECTRL_MATCH1 	|	// related to MATCH1_L
                                                                          SCT_OUTSEL_H			  |	// related to MATCH1_L
                                                                          (AUX_OUT << 6)		  |	// use OUTPUT for I/O condition
                                                                          SCT_IOCOND_HIGH		  |	// AUX = 0
                                                                          SCT_COMBMODE_AND		));	// match AND I/O

	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_7,ENABLE_STATE0);	 			// contains the 12 bits written by the app
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_7, (CHIP_SCT_EVENTCTRL_T)  ( SCT_EVECTRL_MATCH2 	|	// related to MATCH2_L
                                                                          SCT_COMBMODE_MATCH	|	// match only
                                                                          SCT_STATELD_1			  |	// set STATE to a value
                                                                          SCT_STATEEV_9		  ));	// set STATE to 12




	    LPC_SCT->OUT[AUX_OUT].SET  = (1 << 7);               // Event 10 toggles the AUX signal
	    LPC_SCT->OUT[AUX_OUT].CLR  = (1 << 7);               // Event 10 toggles the AUX signal

	    LPC_SCT->OUT[DATA_OUT].SET = (1 << 12)                // Event 15 sets the DATA signal
	                                   | (1 << 9)             // Event 12 sets the DATA signal
	                                   | (1 << 8);            // Event 11 sets the DATA signal
	    LPC_SCT->OUT[DATA_OUT].CLR = (1 << 11)                // Event 14 clears the DATA signal
	                                   | (1 << 10)             // Event 13 clears the DATA signal
	                                   | (1 << 7);            // Event 10 clears the DATA signal


	    Chip_SCT_SetConflictResolution(LPC_SCT, DATA_OUT, 0);	// on conflict: DATA signal doesn't change
	    Chip_SCT_SetConflictResolution(LPC_SCT, AUX_OUT, 3);	// on conflict: AUX signal toggles

	    LPC_SCT->LIMIT_L = (1 << 12);                         	// Event 15 limits the counter

	    LPC_SCT->EVFLAG  = (1 << 7);							// clear event 10 irq flag


	    LPC_SCT->EVEN   |= (1 << 7);							// Event 10 generates an irq

	    NVIC_EnableIRQ(SCT0_IRQn);                             // enable SCT1 interrupt

}

void LEDDRIVER_write(uint32_t rgb)                         // function to write to a transmit buffer
{
    if (LPC_SCT->OUTPUT & (1 << AUX_OUT))                 // aux output determines which buffer to write
    {
    	Chip_SCT_EventState(LPC_SCT , SCT_EVENT_9 , rgb & 0xFFF);
    }
    else
    {
    	Chip_SCT_EventState(LPC_SCT , SCT_EVENT_8 , rgb & 0xFFF);
    }
}

void LEDDRIVER_haltAfterFrame(int on)                      // (de)activate HALT after next frame
{
	 LPC_SCT->HALT_L = (on << 7);                         // if last, event 10 halts the counter
}

void LEDDRIVER_start(void)                                 // start a frame transmission
{
    LPC_SCT->COUNT_L = - WS2811_FB * 50;                  // guarantee 50 µs min time between transfers
    LPC_SCT->STATE_L = 0;                                 // start state
    Chip_SCT_ClearControl(LPC_SCT,SCT_CTRL_HALT_L);	   // unhalt it by clearing bit 2 of the CTRL register                         // start timer H
}



void SCT0_IRQHandler(void)
{

	LPC_SCT->EVFLAG = (1u << 7);       				// clear interrupt flag

    if(LPC_SCT->STATE_L == 9)    						// for debug help
    	Chip_GPIO_SetPinToggle(LPC_GPIO, 1, 0);         // toggle pin P0_7 (LED1)


    if (++count < sizeof(pattern)/sizeof(pattern[0]))  // count nr of frames
    {
        LEDDRIVER_write(pattern[count]);               // write next frame
    }
    else
    {
        LEDDRIVER_haltAfterFrame(1);                   // busy with last frame so halt after this one
    }

    if (LPC_SCT->CTRL_L & 0x04)                       // if halted, transmission completed
    {
        busy = 0;
    }
}

int main(void)
{
	SystemCoreClockUpdate();
	Board_Init();


	Chip_Clock_EnablePeriphClock(	(CHIP_SYSCON_CLOCK_T) ( SYSCON_CLOCK_GPIO1 	|			// enable GPIO port 2 clock
		                                                    SYSCON_CLOCK_IOCON	));		    // enable IOCON clock


	Chip_IOCON_PinMuxSet(LPC_IOCON, 1, 2, IOCON_FUNC3 | IOCON_MODE_INACT | IOCON_DIGITAL_EN | IOCON_INPFILT_OFF);		//SCT0_OUT5 = P1.2  (green   LED)


	Chip_GPIO_SetPortDIROutput(LPC_GPIO, 1, 0);

	LEDDRIVER_open();

    while (1)                                              					// loop forever
    {
    	if (!busy)
    	{
    	    busy  = 1;                                 // transmission pending
    	    count = 0;                                 // used as frame counter
    	    LEDDRIVER_write(pattern[0]);               // preset first data word
    	    LEDDRIVER_haltAfterFrame(0);
    	    LEDDRIVER_start();                         // start transmission
    	 }
    	__WFI();
    }
}
